/**
    Licensed to the Apache Software Foundation (ASF) under one
    or more contributor license agreements.  See the NOTICE file
    distributed with this work for additional information
    regarding copyright ownership.  The ASF licenses this file
    to you under the Apache License, Version 2.0 (the
    "License"); you may not use this file except in compliance
    with the License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing,
    software distributed under the License is distributed on an
    "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
    KIND, either express or implied.  See the License for the
    specific language governing permissions and limitations
    under the License.
*/

/* jshint node:true, bitwise:true, undef:true, trailing:true, quotmark:true,
          indent:4, unused:vars, latedef:nofunc
*/

// The URL:true below prevents jshint error "Redefinition or 'URL'."
/* globals URL:true */

var path          = require('path'),
    fs            = require('fs'),
    shell         = require('shelljs'),
    platforms     = require('./platforms'),
    npmconf       = require('npmconf'),
    events        = require('../events'),
    request       = require('request'),
    config        = require('./config'),
    hooker        = require('./hooker'),
    zlib          = require('zlib'),
    tar           = require('tar'),
    URL           = require('url'),
    Q             = require('q'),
    npm           = require('npm'),
    util          = require('./util');


exports.cordova = cordova;
exports.cordova_git = cordova_git;
exports.cordova_npm = cordova_npm;
exports.custom = custom;
exports.based_on_config = based_on_config;


// Returns a promise for the path to the lazy-loaded directory.
function based_on_config(project_root, platform, opts) {
    var custom_path = config.has_custom_path(project_root, platform);
    if (custom_path) {
        var dot_file = config.read(project_root);
        return module.exports.custom(dot_file.lib[platform].uri, dot_file.lib[platform].id, platform, dot_file.lib[platform].version);
    } else {
        return module.exports.cordova(platform, opts);
    }
}

// Returns a promise for the path to the lazy-loaded directory.
function cordova(platform, opts) {
    var use_npm = opts && opts.usenpm && platform != 'www';
    if ( use_npm ) {
        return module.exports.cordova_npm(platform);
    } else {
        return module.exports.cordova_git(platform);
    }
}

function cordova_git(platform) {
    if (!(platform in platforms)) {
        return Q.reject(new Error('Cordova library "' + platform + '" not recognized.'));
    }

    var url = platforms[platform].url + ';a=snapshot;h=' + platforms[platform].version + ';sf=tgz';
    return module.exports.custom(url, 'cordova', platform, platforms[platform].version);
}

function cordova_npm(platform) {
    var version;
    // Check if platform looks like platform@version
    if (platform.indexOf('@') != -1) {
        var parts = platform.split('@');
        platform = parts[0];
        version = parts[1];
    }
    if ( !(platform in platforms) ) {
        return Q.reject(new Error('Cordova library "' + platform + '" not recognized.'));
    }
    // In most cases platfrom does not specify a version and we use
    // the hard-coded default version from platforms.js
    version = version || platforms[platform].version;
    var pkg = 'cordova-' + platform + '@' + version;
    return Q.nfcall( npm.load, {cache: path.join(util.libDirectory, 'npm_cache') })
    .then(function() {
        return Q.ninvoke(npm.commands, 'cache', ['add', pkg]);
    }).then(function(info) {
        var pkgDir = path.resolve(npm.cache, info.name, info.version, 'package');
        return pkgDir;
    });
}

// Returns a promise for the path to the lazy-loaded directory.
function custom(url, id, platform, version) {
    var download_dir;
    var tmp_dir;
    var lib_dir;

    // Return early for already-cached remote URL, or for local URLs.
    var uri = URL.parse(url);
    var isUri = uri.protocol && uri.protocol[1] != ':'; // second part of conditional is for awesome windows support. fuuu windows
    if (isUri) {
        download_dir = (platform == 'wp8' ? path.join(util.libDirectory, 'wp', id, version) :
                        path.join(util.libDirectory, platform, id, version));
        lib_dir = download_dir;
        if (platforms[platform] && platforms[platform].subdirectory && platform !== 'blackberry10') {
            lib_dir =  path.join(download_dir, platforms[platform].subdirectory);
        }
        if (fs.existsSync(download_dir)) {
            events.emit('verbose', id + ' library for "' + platform + '" already exists. No need to download. Continuing.');
            return Q(lib_dir);
        }
    } else {
        // Local path.
        lib_dir = platforms[platform] && platforms[platform].subdirectory ? path.join(url, platforms[platform].subdirectory) : url;
        return Q(lib_dir);
    }
    return hooker.fire('before_library_download', {
        platform:platform,
        url:url,
        id:id,
        version:version
    }).then(function() {
        var uri = URL.parse(url);
        var d = Q.defer();
        npmconf.load(function(err, conf) {
            // Check if NPM proxy settings are set. If so, include them in the request() call.
            var proxy;
            if (uri.protocol == 'https:') {
                proxy = conf.get('https-proxy');
            } else if (uri.protocol == 'http:') {
                proxy = conf.get('proxy');
            }
            var strictSSL = conf.get('strict-ssl');

            // Create a tmp dir. Using /tmp is a problem because it's often on a different partition and sehll.mv()
            // fails in this case with "EXDEV, cross-device link not permitted".
            var tmp_subidr = 'tmp_' + id + '_' + process.pid + '_' + (new Date()).valueOf();
            tmp_dir = path.join(util.libDirectory, 'tmp', tmp_subidr);
            shell.rm('-rf', tmp_dir);
            shell.mkdir('-p', tmp_dir);

            var size = 0;
            var request_options = {uri:url};
            if (proxy) {
                request_options.proxy = proxy;
            }
            if (typeof strictSSL == 'boolean') {
                request_options.strictSSL = strictSSL;
            }
            events.emit('verbose', 'Requesting ' + JSON.stringify(request_options) + '...');
            events.emit('log', 'Downloading ' + id + ' library for ' + platform + '...');
            var req = request.get(request_options, function(err, res, body) {
                if (err) {
                    shell.rm('-rf', tmp_dir);
                    d.reject(err);
                } else if (res.statusCode != 200) {
                    shell.rm('-rf', tmp_dir);
                    d.reject(new Error('HTTP error ' + res.statusCode + ' retrieving version ' + version + ' of ' + id + ' for ' + platform));
                } else {
                    size = body.length;
                }
            });

            req.pipe(zlib.createUnzip())
            .pipe(tar.Extract({path:tmp_dir}))
            .on('error', function(err) {
                shell.rm('-rf', tmp_dir);
                d.reject(err);
            })
            .on('end', function() {
                events.emit('verbose', 'Downloaded, unzipped and extracted ' + size + ' byte response.');
                events.emit('log', 'Download complete');
                var entries = fs.readdirSync(tmp_dir);
                var entry = path.join(tmp_dir, entries[0]);
                shell.mkdir('-p', download_dir);
                shell.mv('-f', path.join(entry, (platform=='blackberry10'?'blackberry10':''), '*'), download_dir);
                shell.rm('-rf', tmp_dir);
                d.resolve(hooker.fire('after_library_download', {
                    platform:platform,
                    url:url,
                    id:id,
                    version:version,
                    path: lib_dir,
                    size:size,
                    symlink:false
                }));
            });
        });
        return d.promise.then(function () { return lib_dir; });
    });
}


